# 機能設計書 26-cuDNN RNN

## 概要

本ドキュメントは、TensorFlowにおけるGPU向け最適化RNN実装（cuDNN LSTM・GRU）の機能設計を記述する。NVIDIA cuDNNライブラリを活用してRNN計算を高速化するOp群を提供する。

### 本機能の処理概要

**業務上の目的・背景**：標準的なRNN実装はタイムステップごとの逐次処理により計算コストが高い。cuDNN RNNは、NVIDIA GPU上で高度に最適化されたカーネルを使用することで、LSTM・GRU・SimpleRNNの学習・推論を数倍高速化する。大規模な自然言語処理・音声認識モデルの学習時間を大幅に短縮する。

**機能の利用シーン**：NVIDIA GPU搭載環境での大規模RNNモデルの学習・推論。特に長いシーケンスや大きなバッチサイズでの学習時に効果が大きい。Kerasの`tf.keras.layers.LSTM`や`tf.keras.layers.GRU`が特定条件下で自動的にcuDNNカーネルを選択する。

**主要な処理内容**：
1. CudnnRNNParamsSize: cuDNN RNNに必要なパラメータサイズの計算
2. CudnnRNN / CudnnRNNV2 / CudnnRNNV3: 順方向RNN計算
3. CudnnRNNBackprop / CudnnRNNBackpropV2 / CudnnRNNBackpropV3: 逆方向（勾配）計算
4. CudnnRNNCanonicalToParams / CudnnRNNParamsToCanonical: パラメータ形式変換
5. 対応RNNモード: rnn_relu, rnn_tanh, lstm, gru
6. 単方向・双方向の両方をサポート

**関連システム・外部連携**：リカレント層（No.21）のKeras LSTM/GRUレイヤーから自動的に呼び出される。NVIDIA cuDNNライブラリ（GPU専用）に依存。

**権限による制御**：特になし。GPU環境でのみ利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングに該当なし |

## 機能種別

計算処理（GPU最適化テンソル演算）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| input | Tensor(T) | Yes | 入力テンソル [seq_length, batch_size, input_size] | 3次元 |
| input_h | Tensor(T) | Yes | 初期隠れ状態 [num_layers*dir, batch_size, num_units] | 3次元 |
| input_c | Tensor(T) | Yes | 初期セル状態（LSTM用） | 3次元（LSTMのみ） |
| params | Tensor(T) | Yes | パラメータ（1D flatten） | 1次元 |
| rnn_mode | string | No | RNNモード | 'rnn_relu','rnn_tanh','lstm','gru'、デフォルト:'lstm' |
| input_mode | string | No | 入力モード | 'linear_input','skip_input','auto_select' |
| direction | string | No | 方向 | 'unidirectional','bidirectional' |
| dropout | float | No | ドロップアウト率 | デフォルト: 0.0 |
| seed/seed2 | int | No | 乱数シード | デフォルト: 0 |
| is_training | bool | No | 学習モードか | デフォルト: true |
| T | dtype | Yes | データ型 | bfloat16, float16, float32, float64 |

### 入力データソース

Keras RNNレイヤーからの入力テンソル。パラメータはカーネル・リカレントカーネル・バイアスを1D flattenしたもの。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| output | Tensor(T) | RNN出力 [seq_length, batch_size, num_units * dir_count] |
| output_h | Tensor(T) | 最終隠れ状態 |
| output_c | Tensor(T) | 最終セル状態（LSTMのみ、GRU等は空） |
| reserve_space | Tensor(T) | 逆伝播用の予約領域 |
| host_reserved | Tensor(int8) | ホスト側予約領域（V2以降） |

### 出力先

Keras RNNレイヤーへの出力。reserve_spaceは逆伝播時に使用。

## 処理フロー

### 処理シーケンス

```
1. パラメータサイズ計算（CudnnRNNParamsSize）
   └─ num_layers, num_units, input_sizeからパラメータ総数を算出
2. 順方向計算（CudnnRNN/V2/V3）
   └─ cuDNNカーネルによるRNN順方向処理
   └─ output, output_h, output_c, reserve_spaceを出力
3. 逆方向計算（CudnnRNNBackprop/V2/V3）
   └─ reserve_spaceを使用して勾配計算
4. パラメータ変換（必要に応じて）
   └─ Canonical形式 ↔ cuDNN形式の相互変換
```

### フローチャート

```mermaid
flowchart TD
    A[入力: input, input_h, input_c, params] --> B[CudnnRNNParamsSize]
    B --> C[パラメータ割り当て]
    C --> D[CudnnRNN Forward]
    D --> E[output, output_h, output_c]
    D --> F[reserve_space]
    E --> G[Keras RNN層出力]
    F --> H[CudnnRNNBackprop]
    H --> I[勾配: d_input, d_input_h, d_input_c, d_params]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-26-01 | GPU専用 | cuDNN RNNはNVIDIA GPU上でのみ実行可能 | デバイス選択時 |
| BR-26-02 | 双方向出力サイズ | bidirectionalの場合、出力はnum_units*2 | direction='bidirectional' |
| BR-26-03 | LSTM専用セル状態 | input_c/output_cはLSTMモードでのみ意味を持つ | rnn_mode='lstm' |
| BR-26-04 | num_proj | 射影サイズ指定でLSTMの出力次元を変更可能 | V3以降 |

### 計算ロジック

cuDNN内部の最適化カーネルにより、タイムステップの並列化とメモリアクセスの最適化が行われる。具体的な計算はcuDNNライブラリに委譲。

## データベース操作仕様

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| InvalidArgument | 形状エラー | 入力が3次元でない | 正しい形状のテンソルを渡す |
| Unimplemented | デバイスエラー | CPUでの実行を試みた | GPU環境で実行する |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- 標準RNN実装と比較して2-10倍の高速化が期待される
- reserve_spaceによるメモリ使用量の増加あり（逆伝播用）
- bfloat16, float16サポートによるTensorコア活用が可能

## セキュリティ考慮事項

特になし。

## 備考

- ステートフル（SetIsStateful）: ドロップアウト使用時のランダム状態管理
- V1→V2→V3と進化し、V3ではnum_proj（射影サイズ）をサポート
- Kerasレイヤーは条件（デフォルト活性化関数、recurrent_dropout=0等）を満たすと自動的にcuDNNカーネルを選択

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: Op定義（定数と属性）

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | cudnn_rnn_ops.cc | `tensorflow/core/ops/cudnn_rnn_ops.cc` | 24-33行目のkRNNModeAttrs, kRNNInputModeAttrs, kRNNDirectionAttrs定数定義 |

**読解のコツ**: 全Opで共通の3つの属性（rnn_mode, input_mode, direction）が`constexpr`で定義されている。これらがcuDNNの動作モードを制御する。

#### Step 2: 主要Op定義

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | cudnn_rnn_ops.cc | `tensorflow/core/ops/cudnn_rnn_ops.cc` | CudnnRNNParamsSize（40-63行目）: パラメータサイズ算出Op |
| 2-2 | cudnn_rnn_ops.cc | `tensorflow/core/ops/cudnn_rnn_ops.cc` | CudnnRNN（65-111行目）: 順方向計算Op |
| 2-3 | cudnn_rnn_ops.cc | `tensorflow/core/ops/cudnn_rnn_ops.cc` | CudnnRNNV2（113-150行目）: V2順方向計算Op（host_reserved追加） |

**主要処理フロー（CudnnRNN形状推論）**:
- **87-89行目**: 入力形状の検証（input: 3D, input_h: 3D, params: 1D）
- **91-93行目**: seq_length, batch_size, num_unitsの抽出
- **96-99行目**: direction属性の取得とdir_count計算
- **100-101行目**: 出力サイズ = num_units * dir_count
- **102-109行目**: output, output_h, output_c, reserve_space形状の設定

### プログラム呼び出し階層図

```
tf.keras.layers.LSTM / GRU (自動cuDNN選択)
    │
    ├─ CudnnRNNParamsSize
    │      └─ パラメータ総サイズ算出
    │
    ├─ CudnnRNN / V2 / V3 (順方向)
    │      ├─ input [T,B,I] → output [T,B,H*dir]
    │      ├─ input_h → output_h
    │      ├─ input_c → output_c (LSTM only)
    │      └─ → reserve_space (for backprop)
    │
    ├─ CudnnRNNBackprop / V2 / V3 (逆方向)
    │      └─ reserve_space → gradients
    │
    └─ CudnnRNNCanonicalToParams / ParamsToCanonical
           └─ パラメータ形式変換
```

### データフロー図

```
[入力]                            [処理]                         [出力]

input [T,B,I]           ───▶  CudnnRNN Forward          ───▶  output [T,B,H*dir]
input_h [L*dir,B,H]     ───▶     (cuDNN kernel)         ───▶  output_h [L*dir,B,H]
input_c [L*dir,B,H]     ───▶                            ───▶  output_c (LSTM)
params [param_size]      ───▶                            ───▶  reserve_space

d_output                 ───▶  CudnnRNNBackprop          ───▶  d_input, d_input_h
d_output_h               ───▶     (cuDNN kernel)         ───▶  d_input_c, d_params
reserve_space            ───▶
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| cudnn_rnn_ops.cc | `tensorflow/core/ops/cudnn_rnn_ops.cc` | ソース | cuDNN RNN Op定義 |
| rnn_ops.cc | `tensorflow/core/ops/rnn_ops.cc` | ソース | GRUBlockCell/LSTMBlockCell Op |
| recurrent.py | `tensorflow/python/keras/layers/recurrent.py` | ソース | Keras RNN層（cuDNN自動選択） |
